<?php

namespace App\Http\Controllers;

use App\Models\Order;
use App\Models\OrderItem;
use App\Models\Product;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class CheckoutController extends Controller
{
    /**
     * ============================
     * HALAMAN CHECKOUT
     * ============================
     */
    public function index()
    {
        $checkoutCart = session('checkout_cart', []);

        if (empty($checkoutCart)) {
            return redirect()->route('home')
                ->with('error', 'Tidak ada produk untuk checkout');
        }

        $subtotal = collect($checkoutCart)
            ->sum(fn ($item) => $item['price'] * $item['qty']);

        // ❗ ongkir hanya placeholder di UI
        $ongkir = 0;
        $total  = $subtotal;

        return view('checkout', compact(
            'checkoutCart',
            'subtotal',
            'ongkir',
            'total'
        ));
    }

    /**
     * ============================
     * ORDER NOW
     * ============================
     */
    public function orderNow($productId)
    {
        $product = Product::findOrFail($productId);

        session()->forget('checkout_cart');

        session([
            'checkout_cart' => [
                $product->id => [
                    'product_id' => $product->id,
                    'name'       => $product->name,
                    'price'      => $product->price,
                    'qty'        => 1,
                    'image'      => $product->image,
                ]
            ]
        ]);

        return redirect()->route('checkout.index');
    }

    /**
     * ============================
     * UPDATE QTY
     * ============================
     */
    public function updateQty(Request $request, $productId)
    {
        $checkoutCart = session('checkout_cart', []);

        if (!isset($checkoutCart[$productId])) {
            return response()->json(['error' => true], 404);
        }

        $checkoutCart[$productId]['qty'] = max(1, (int) $request->qty);
        session(['checkout_cart' => $checkoutCart]);

        $subtotal = collect($checkoutCart)
            ->sum(fn ($item) => $item['price'] * $item['qty']);

        return response()->json([
            'qty'      => $checkoutCart[$productId]['qty'],
            'subtotal' => $subtotal,   // angka murni
            'total'    => $subtotal,   // ongkir dihitung di frontend
        ]);
    }

    /**
     * ============================
     * BUAT ORDER (MANUAL PAYMENT)
     * ============================
     */
    public function pay(Request $request)
    {
        $request->validate([
            'receiver_name'   => 'required|string|max:100',
            'address'         => 'required|string',
            'phone'           => ['required','regex:/^(08|\+628)[0-9]{8,11}$/'],
            'payment_channel' => 'required|in:bank,ewallet',
            'payment_name'    => 'required|string',

            // 🔥 VALIDASI ONGKIR
            'shipping_cost'   => 'required|integer|min:0',
            'total_price'     => 'required|integer|min:0',
        ]);

        $checkoutCart = session('checkout_cart', []);
        if (empty($checkoutCart)) {
            return redirect()->route('home');
        }

        $subtotal = collect($checkoutCart)
            ->sum(fn ($item) => $item['price'] * $item['qty']);

        // 🔥 AMBIL ONGKIR DARI REQUEST (HASIL BITESHIP)
        $ongkir = (int) $request->shipping_cost;
        $total  = $subtotal + $ongkir;

        // 🔐 SAFETY CHECK
        if ($total !== (int) $request->total_price) {
            abort(422, 'Total tidak valid');
        }

        $order = Order::create([
            'user_id'         => Auth::id(),
            'receiver_name'   => $request->receiver_name,
            'address'        => $request->full_address,
            'phone'           => $request->phone,

            'courier'         => $request->courier,
            'courier_service' => $request->courier_service,

            'shipping_cost'   => $ongkir,
            'total_price'     => $total,

            'payment_channel' => $request->payment_channel,
            'payment_name'    => $request->payment_name,

            'method'          => 'manual',
            'status'          => 'pending',
        ]);

       foreach ($checkoutCart as $productId => $item) {
    OrderItem::create([
        'order_id'   => $order->id,
        'product_id' => $productId,   // 🔥 AMBIL DARI KEY
        'qty'        => $item['qty'],
        'price'      => $item['price'],
    ]);
}


        session()->forget('checkout_cart');

        return redirect()->route('payment.manual', $order);
    }

    /**
     * ============================
     * HALAMAN PEMBAYARAN
     * ============================
     */
    public function manual(Order $order)
{
    if ($order->user_id !== Auth::id()) {
        abort(403);
    }

    $paymentTargets = [
        'BRI' => [
            'number' => '641701043787531',
            'name'   => 'Kopi Ketjeh'
        ],
        'DANA' => [
            'number' => '085859857140',
            'name'   => 'Kopi Ketjeh'
        ],
    ];

    // 🔥 PAKAI payment_name (BRI / DANA)
    if (
        !empty($order->payment_name) &&
        empty($order->payment_number) &&
        isset($paymentTargets[$order->payment_name])
    ) {
        $order->update([
            'payment_number' => $paymentTargets[$order->payment_name]['number'],
        ]);
    }

    return view('payment.manual', compact('order', 'paymentTargets'));
}


    /**
     * ============================
     * UPLOAD BUKTI PEMBAYARAN
     * ============================
     */
   public function uploadProof(Request $request, Order $order)
{
    if ($order->user_id !== Auth::id()) {
        abort(403);
    }

    $request->validate([
        'payment_proof' => 'required|image|mimes:jpg,jpeg,png|max:2048',
    ]);

    $filename = time().'.'.$request->file('payment_proof')->extension();

    $path = $request->file('payment_proof')
        ->storeAs('payment_proofs', $filename, 'public');

    // 🔥 DAFTAR NOMOR TUJUAN (SUMBER RESMI)
    $paymentTargets = [
        'BRI'  => '641701043787531',
        'DANA' => '085859857140',
    ];

    $data = [
        'payment_proof' => $path,
        'status'        => 'waiting_verification',
    ];

    // 🔥 SIMPAN NOMOR TUJUAN (INI KUNCI UTAMA)
    if (
        !empty($order->payment_name) &&
        empty($order->payment_number) &&
        isset($paymentTargets[$order->payment_name])
    ) {
        $data['payment_number'] = $paymentTargets[$order->payment_name];
    }

    $order->update($data);

    return redirect()->route('akun')
        ->with('success', 'Bukti pembayaran berhasil dikirim, menunggu verifikasi admin');
}

}